from docplex.mp.model import Model
import json
import sys

# TRANSPORTATION MODEL: additional features

###DATA (read from a file, Load the JSON file)
with open(sys.argv[1] if len(sys.argv)>1 else 'transport_plus.json', 'r') as file:
    data = json.load(file)

modelname = data["modelname"]
#read sets
I = data["I"] # given as ordered list of origins
J = data["J"] # given as ordered list of destinations
#read parameters
o_list = data["o_list"] # ORDERED list of the origins capacity
d_list = data["d_list"] # ORDERED list of the destinations requests
c_matrix = data["c_matrix"] # o/d cost matrix based on the order of orgines and destinations
decision_domain =  data["decision_domain"]
cost_threshold_percent = data["cost_threshold_percent"]

####DATA-INDEPENDENT MODEL CREATION
# data preprocessing (transform ordered list into dictionaries, for ease of reference)
O = {}
for i in range(len(I)):
    O[I[i]] = o_list[i] # capacity of each origin 

D = {J[j]: d_list[j] for j in range(len(J))} # request of each destination

C = { (I[i],J[j]): c_matrix[i][j] for i in range(len(I)) for j in range(len(J))} # transport cost of a unit from origin i to destination j

# model definition
m = Model(name = "transportation")

# define the set of "non expensive" accepted links
ActiveODpairs =  [(i, j) for i in I for j in J 
                  if C[(i,j)] <= cost_threshold_percent * max(C[(o,d)] for o in O for d in D)]

# define variables only for accepted links (cannot use "matrix")
if decision_domain == "discrete":
    x = m.integer_var_dict( keys = ActiveODpairs,lb = 0, ub = None, name = 'x')
else:
    x = m.continuous_var_dict( keys = ActiveODpairs,lb = 0, ub = None, name = 'x')

m.minimize(m.sum(C[(i,j)]*x[(i,j)] for (i,j) in ActiveODpairs))

#constraints: basic
for j in J:
    m.add_constraint(m.sum(x[(i,j)] for i in I if (i,j) in ActiveODpairs )>= D[j])

m.add_constraints(m.sum(x[(ii,jj)] for (ii,jj) in ActiveODpairs if ii==i )<= O[i] for i in I)

print("SOLVING THE BASIC MODEL")
m.print_information()
sol = m.solve(log_output = False)
if sol:
    m.print_solution()
    store_sol_val = m.objective_value
else:
    print("Basic model: no solution availabe",m.get_solve_status())


print("CONSIDERING ADDITIONAL CONSTRAINTS")
LowCost = data["LowCost"]
LowCostMinOnLink = data["LowCostMinOnLink"]
#constraint: additional 1 (see slides)
m.add_constraints( x[(i,j)] >= LowCostMinOnLink for (i,j) in ActiveODpairs if C[(i,j)] <= LowCost)

#constraint: additional 2 (see slides)
SpecialDestination = data["SpecialDestination"]
MinToSpecialDestination = data["MinToSpecialDestination"]
SpecialOrigin = data["SpecialOrigin"] 
m.add_constraints( x[(i,SpecialDestination)] >= MinToSpecialDestination for i in I if i != SpecialOrigin and (i,SpecialDestination) in ActiveODpairs )
# or, equivalently
# m.add_constraints( x[(i,j)] >= MinToSpecialDestination for (i,j) in ActiveODpairs if i != SpecialOrigin and j == SpecialDestination )

#constraint: additional 3 (see slides)
SignificantNumber = data["SignificantNumber"]
SignificantFraction = data["SignificantFraction"]
y = m.binary_var_dict(ActiveODpairs, name="y")
for j in J: 
    m.add_constraint(m.sum(y[od] for od in ActiveODpairs if od[1]==j) >= SignificantNumber)
m.add_constraints(x[od] >= SignificantFraction*D[od[1]] * y[od] for od in ActiveODpairs) #"activate" y variables

m.print_information() #display information on the model 
m.export_as_lp(basename='transport_plus', path = '.', hide_user_names=False) # export the model in the given path in LP format

sol = m.solve(log_output = False)
if sol:
    #customized output
    print("The new total cost is ", '{0:.2f}'.format(sol.get_objective_value()), ": ", 
            "{0:.2f}".format(sol.get_objective_value()-store_sol_val), " more than basic")
    for (i,j) in ActiveODpairs:
        if sol[x[(i,j)]] > 0:
            print((i,j), ":", sol[x[(i,j)]], "(small, up to %4.1f%% of %s's demand)" % (SignificantFraction*100, j) if sol[y[(i,j)]]<=0.001 else "")
    #standard print solution
    m.print_solution()    #display information on the solution
    m.solution.export('transport_solution.json') # exports solution in a json file
else:
    print("Status after solving with additional constraints ",m.get_solve_status())



