/********************************************************************************************************************
 * \file MCMCAN.c
 * \copyright Copyright (C) Infineon Technologies AG 2020
 *
 * Use of this file is subject to the terms of use agreed between (i) you or the company in which ordinary course of
 * business you are acting and (ii) Infineon Technologies AG or its licensees. If and as long as no such terms of use
 * are agreed, use of this file is subject to following:
 *
 * Boost Software License - Version 1.0 - August 17th, 2003
 *
 * Permission is hereby granted, free of charge, to any person or organization obtaining a copy of the software and
 * accompanying documentation covered by this license (the "Software") to use, reproduce, display, distribute, execute,
 * and transmit the Software, and to prepare derivative works of the Software, and to permit third-parties to whom the
 * Software is furnished to do so, all subject to the following:
 *
 * The copyright notices in the Software and this entire statement, including the above license grant, this restriction
 * and the following disclaimer, must be included in all copies of the Software, in whole or in part, and all
 * derivative works of the Software, unless such copies or derivative works are solely in the form of
 * machine-executable object code generated by a source language processor.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 *********************************************************************************************************************/

/*********************************************************************************************************************/
/*-----------------------------------------------------Includes------------------------------------------------------*/
/*********************************************************************************************************************/
#include "MCMCAN.h"

/*********************************************************************************************************************/
/*-------------------------------------------------Global variables--------------------------------------------------*/
/*********************************************************************************************************************/
McmcanType                  g_mcmcan;                       /* Global MCMCAN configuration and control structure    */
IfxPort_Pin_Config          g_led1;                         /* Global LED1 configuration and control structure      */
IfxPort_Pin_Config          g_led2;                         /* Global LED2 configuration and control structure      */

/*********************************************************************************************************************/
/*---------------------------------------------Function Implementations----------------------------------------------*/
/*********************************************************************************************************************/
/* Macro to define Interrupt Service Routine.
 * This macro:
 * - defines linker section as .intvec_tc<vector number>_<interrupt priority>.
 * - defines compiler specific attribute for the interrupt functions.
 * - defines the Interrupt service routine as ISR function.
 *
 * IFX_INTERRUPT(isr, vectabNum, priority)
 *  - isr: Name of the ISR function.
 *  - vectabNum: Vector table number.
 *  - priority: Interrupt priority. Refer Usage of Interrupt Macro for more details.
 */
IFX_INTERRUPT(canIsrTxHandler, 0, ISR_PRIORITY_CAN_TX);
IFX_INTERRUPT(canIsrRxHandler, 0, ISR_PRIORITY_CAN_RX);

/* Interrupt Service Routine (ISR) called once the TX interrupt has been generated.
 * Turns on the LED1 to indicate successful CAN message transmission.
 */
void canIsrTxHandler(void)
{
    /* Clear the "Transmission Completed" interrupt flag */
    IfxCan_Node_clearInterruptFlag(g_mcmcan.canNode.node, IfxCan_Interrupt_transmissionCompleted);

    /* Just to indicate that the CAN message has been transmitted by turning on LED1 */
    IfxPort_setPinLow(g_led1.port, g_led1.pinIndex);
}

/* Interrupt Service Routine (ISR) called once the RX interrupt has been generated.
 * Compares the content of the received CAN message with the content of the transmitted CAN message
 * and in case of success, turns on the LED2 to indicate successful CAN message reception.
 */
void canIsrRxHandler(void)
{
    /* Clear the "Message stored to Dedicated RX Buffer" interrupt flag */
    IfxCan_Node_clearInterruptFlag(g_mcmcan.canNode.node, IfxCan_Interrupt_messageStoredToDedicatedRxBuffer);

    /* Read the received CAN message */
    IfxCan_Can_readMessage(&g_mcmcan.canNode, &g_mcmcan.rxMsg, g_mcmcan.rxData);

    /* Check if the received data matches with the transmitted one */
    if( ( g_mcmcan.rxData[0] == g_mcmcan.txData[0] ) &&
        ( g_mcmcan.rxData[1] == g_mcmcan.txData[1] ) &&
        ( g_mcmcan.rxMsg.messageId == g_mcmcan.txMsg.messageId ) )
    {
        /* Turn on the LED2 to indicate correctness of the received message */
        IfxPort_setPinLow(g_led2.port, g_led2.pinIndex);
    }
}

/* Function to initialize MCMCAN module and nodes related for this application use case */
void initMcmcan(void)
{
    IfxCan_Can_Pins canPins;

    /* ==========================================================================================
     * CAN module configuration and initialization:
     * ==========================================================================================
     *  - load default CAN module configuration into configuration structure
     *  - initialize CAN module with the default configuration
     * ==========================================================================================
     */
    
    IfxCan_Can_initModuleConfig(&g_mcmcan.canConfig, &MODULE_CAN0);
    // The previous row can be replaced with -> g_mcmcan.canConfig.can = &MODULE_CAN0;

    IfxCan_Can_initModule(&g_mcmcan.canModule, &g_mcmcan.canConfig);
    /*
    * The previous row enable the CAN module 0 and can be replaced with -> CLC.B.DISR = 0U;
    * BUT DON'T DO IT!! This is a "sensible" register and is protected by some 
    * hardware mechanisms that prevent its accidental writing.
    */

    /* ==========================================================================================
     * Source CAN node configuration and initialization:
     * ==========================================================================================
     *  - load default CAN node configuration into configuration structure
     *
     *  - assign source CAN node to CAN node 0
     *
     *  - set baudrate to 1Mbps
     *
     *  - define the frame to be the transmitting one
     *
     *  - once the transmission is completed, raise the interrupt
     *  - define the transmission complete interrupt priority
     *  - assign the interrupt line 0 to the transmission complete interrupt
     *  - transmission complete interrupt service routine should be serviced by the CPU0
     * 
     *  - once the message is stored in the dedicated RX buffer, raise the interrupt
     *  - define the receive interrupt priority
     *  - assign the interrupt line 1 to the receive interrupt
     *  - receive interrupt service routine should be serviced by the CPU0
     *
     *  - initialize the source CAN node with the modified configuration
     * ==========================================================================================
     */
    IfxCan_Can_initNodeConfig(&g_mcmcan.canNodeConfig, &g_mcmcan.canModule);

    // We don't need to communicate in loopback mode with another (internal) CAN node...
    // So we can delete the loop enabling
    // g_mcmcan.canNodeConfig.busLoopbackEnabled = TRUE;
    g_mcmcan.canNodeConfig.nodeId = IfxCan_NodeId_0;
    g_mcmcan.canNodeConfig.baudRate.baudrate = 1000000;
    // We need to change the type frame. 
    // In the loopback example, we had two node: a TX one and a RX one.
    // The frame type of the TX node was IfxCan_FrameType_transmit, while 
    // the one of the RX node was IfxCan_FrameType_receive.
    // Which is the needed one for a node is able of both TX and RX?
    // Look at the IfxCan_FrameType...
    g_mcmcan.canNodeConfig.frame.type = IfxCan_FrameType_transmitAndReceive;

    g_mcmcan.canNodeConfig.interruptConfig.transmissionCompletedEnabled = TRUE;
    g_mcmcan.canNodeConfig.interruptConfig.traco.priority = ISR_PRIORITY_CAN_TX;
    g_mcmcan.canNodeConfig.interruptConfig.traco.interruptLine = IfxCan_InterruptLine_0;
    g_mcmcan.canNodeConfig.interruptConfig.traco.typeOfService = IfxSrc_Tos_cpu0;

    g_mcmcan.canNodeConfig.interruptConfig.messageStoredToDedicatedRxBufferEnabled = TRUE;
    g_mcmcan.canNodeConfig.interruptConfig.reint.priority = ISR_PRIORITY_CAN_RX;
    g_mcmcan.canNodeConfig.interruptConfig.reint.interruptLine = IfxCan_InterruptLine_1;
    g_mcmcan.canNodeConfig.interruptConfig.reint.typeOfService = IfxSrc_Tos_cpu0;

    /* ------------------------------------------------------------------------------------------
     * Pin configuration
     * ------------------------------------------------------------------------------------------
     * - Define the Rx pin based on connection to CAN PHY (check the Lite-Kit manual or schematic)
     * - Define the Rx pin in input pull-up mode
     * - Define the Tx pin based on connection to CAN PHY (check the Lite-Kit manual or schematic)
     * - Define the Tx pin in output push-pull mode
     * - Define the speed grade of Tx pin
     * ------------------------------------------------------------------------------------------
     */

    // We need to configure the PIN 20.7 with the correct "input selection".
    // You need to: 
    //  - dive into the manuals (LiteKite UM, TC37x datasheet, TC37x UM - can connections)
    //  - look at IfxCan_Rxd_In type
    canPins.rxPin = &IfxCan_RXD00B_P20_7_IN;
    canPins.rxPinMode = IfxPort_InputMode_pullUp;
    // We need to configure the PIN 20.8 with the correct "input selection".
    // You need to: 
    //  - dive into the manuals (LiteKite UM, TC37x datasheet, TC37x UM - can connections)
    //  - look at IfxCan_Txd_Out type
    canPins.txPin = &IfxCan_TXD00_P20_8_OUT;
    canPins.txPinMode = IfxPort_OutputMode_pushPull;

    canPins.padDriver = IfxPort_PadDriver_cmosAutomotiveSpeed3;
    g_mcmcan.canNodeConfig.pins = &canPins;

    IfxCan_Can_initNode(&g_mcmcan.canNode, &g_mcmcan.canNodeConfig);

    /* ==========================================================================================
     * CAN filter configuration and initialization:
     * ==========================================================================================
     *  - filter configuration is stored under the filter element number 0
     *  - store received frame in a dedicated RX Buffer
     *  - define the same message ID as defined for the TX message
     *  - assign the filter to the dedicated RX Buffer (RxBuffer0 in this case)
     *
     *  - initialize the standard filter with the modified configuration
     * ==========================================================================================
     */
    g_mcmcan.canFilter.number = 0;
    g_mcmcan.canFilter.elementConfiguration = IfxCan_FilterElementConfiguration_storeInRxBuffer;
    g_mcmcan.canFilter.id1 = CAN_MESSAGE_ID;
    g_mcmcan.canFilter.rxBufferOffset = IfxCan_RxBufferId_0;

    IfxCan_Can_setStandardFilter(&g_mcmcan.canNode, &g_mcmcan.canFilter);
}

/* Function to initialize both TX and RX messages with the default data values.
 * After initialization of the messages, the TX message is transmitted.
 */
void transmitCanMessage(void)
{
    /* Initialization of the RX message with the default configuration */
    IfxCan_Can_initMessage(&g_mcmcan.rxMsg);

    /* Invalidation of the RX message data content */
    memset((void *)(&g_mcmcan.rxData[0]), INVALID_RX_DATA_VALUE, MAXIMUM_CAN_DATA_PAYLOAD * sizeof(uint32));

    /* Initialization of the TX message with the default configuration */
    IfxCan_Can_initMessage(&g_mcmcan.txMsg);

    /* Define the content of the data to be transmitted */
    g_mcmcan.txData[0] = TX_DATA_LOW_WORD;
    g_mcmcan.txData[1] = TX_DATA_HIGH_WORD;

    /* Set the message ID that is used during the receive acceptance phase */
    g_mcmcan.txMsg.messageId = CAN_MESSAGE_ID;

    /* Send the CAN message with the previously defined TX message content */
    while( IfxCan_Status_notSentBusy ==
           IfxCan_Can_sendMessage(&g_mcmcan.canNode, &g_mcmcan.txMsg, &g_mcmcan.txData[0]) )
    {
    }
}

/* Function to initialize the LEDs */
void initLeds(void)
{
    /* ======================================================================
     * Configuration of the pins connected to the LEDs:
     * ======================================================================
     *  - define the GPIO port
     *  - define the GPIO pin that is connected to the LED
     *  - define the general GPIO pin usage (no alternate function used)
     *  - define the pad driver strength
     * ======================================================================
     */
    g_led1.port      = &MODULE_P00;
    g_led1.pinIndex  = PIN5;
    g_led1.mode      = IfxPort_OutputIdx_general;
    g_led1.padDriver = IfxPort_PadDriver_cmosAutomotiveSpeed1;

    g_led2.port      = &MODULE_P00;
    g_led2.pinIndex  = PIN6;
    g_led2.mode      = IfxPort_OutputIdx_general;
    g_led2.padDriver = IfxPort_PadDriver_cmosAutomotiveSpeed1;

    /* Initialize the pins connected to LEDs to level "HIGH", which keep the LEDs turned off as default state */
    IfxPort_setPinHigh(g_led1.port, g_led1.pinIndex);
    IfxPort_setPinHigh(g_led2.port, g_led2.pinIndex);

    /* Set the pin input/output mode for both pins connected to the LEDs */
    IfxPort_setPinModeOutput(g_led1.port, g_led1.pinIndex, IfxPort_OutputMode_pushPull, g_led1.mode);
    IfxPort_setPinModeOutput(g_led2.port, g_led2.pinIndex, IfxPort_OutputMode_pushPull, g_led2.mode);

    /* Set the pad driver mode for both pins connected to the LEDs */
    IfxPort_setPinPadDriver(g_led1.port, g_led1.pinIndex, g_led1.padDriver);
    IfxPort_setPinPadDriver(g_led2.port, g_led2.pinIndex, g_led2.padDriver);
}

/* Function to initialize the CAN_STB pin */
void initCanStbPin(void)
{
    IfxPort_Pin_Config stbPin;

    stbPin.port      = &MODULE_P20;
    stbPin.pinIndex  = PIN6;
    stbPin.mode      = IfxPort_OutputIdx_general;
    stbPin.padDriver = IfxPort_PadDriver_cmosAutomotiveSpeed1;

    /* Initialize the pin connected to CAN_STB to level "LOW" for Normal-operating mode */
    // I have to put the pin low... which function of the iLLD help me?
    // Look at IfxPort.h and IfxPort_reg.h
    // INSERT THE FUNCTION HERE
    IfxPort_setPinLow(stbPin.port, stbPin.pinIndex);
    //

    /* Set the pin output mode for pin connected to the CAN_STB */
    // I have to configure P20.6 as output mode... which function of the iLLD help me?
    // Look at IfxPort.h and IfxPort_reg.h
    // INSERT THE FUNCTION HERE
    IfxPort_setPinModeOutput(stbPin.port, stbPin.pinIndex, IfxPort_OutputMode_pushPull, stbPin.mode);
    //

    /* Set the pad driver mode for pin connected to the CAN_STB */
    // Look at IfxPort.h and IfxPort_reg.h
    // INSERT THE FUNCTION HERE
    IfxPort_setPinPadDriver(stbPin.port, stbPin.pinIndex, stbPin.padDriver);
    //
}
