from docplex.mp.model import Model
from random import random

''' INSTANCE DATA
for the sake of this example, we create "simple" data structures 
and populate them directly in this code with random data.
You can read data from any source, using your favourite python library
'''
#sets
num_origins = 3
num_destinations = 4
I = range(0,num_origins)      #I={0,1,2}
J = range(0,num_destinations) #J={0,1,2,3}
#parameters
cost = {(i,j): random()*10 for i in I for j in J}
origin_capacity = [int(random()*100) for i in I]
destination_request = [int(random()*100) for j in J]
origin_capacity[0] += max(0,sum(destination_request) - sum(origin_capacity)) #balanced problem, to guarantee feasibility
print(origin_capacity)
print(destination_request)

''' MODEL DEFINITION
variables and constraints are created using python iterators (e.g., "for")
to replicate the algebraic indexing expressions appearing 
into the "forall" quantifiers and into sums
'''
#create an empty model
m = Model(name="transportation")
#create variables 'x' as a python dictionary of docplex <type>_var indexed by (i,j) pairs,
# using a python iterator (forall quantifier in the model)
x = {(i,j): m.continuous_var(name='x from {0} to {1}'.format(i,j)) for i in I for j in J}
#create the objective function using docplex "sum" and python iterator (sum indexes in the model)
m.minimize(m.sum(cost[i,j] * x[i,j] for i in I for j in J))
#create basic contraints (one at a time in loops)
for j in J:
    m.add_constraint(   m.sum(x[i,j] for i in I) >= destination_request[j]   )
for i in I:
    m.add_constraint(   m.sum(x[i,j] for j in J) <= origin_capacity[i]       )

m.print_information()
sol = m.solve(log_output = False)
m.print_solution()
store_sol_val=m.objective_value

print("\nADDING FURTHER CONSTRAINTS")
#at least two origins significantly supply each destination
significant_fraction = 0.25
#alternative (i,j) var definition using docplex ...var_dict
ODpairs = [(i, j) for i in I for j in J]
y = m.binary_var_dict(ODpairs , name="y")
for j in J:
    m.add_constraint(m.sum(y[od] for od in ODpairs if od[1]==j) >= 2)
#defining a group of constraints (python iterator == forall in the model)
m.add_constraints(x[od] >= significant_fraction*destination_request[od[1]] * y[od] for od in ODpairs)
m.print_information()

#solve with further constraints (may be unfeasible)
sol = m.solve(log_output = True)
if sol:
    #customized output
    print("Solution with further constraints:")
    print("The new total cost is ", '{0:.2f}'.format(sol.get_objective_value()), ": ", 
            "{0:.2f}".format(sol.get_objective_value()-store_sol_val), " more than base")
    for od in ODpairs:
        if sol[x[od]] > 0:
            print(od, ":", sol[x[od]], sol[y[od]], "(small)" if sol[y[od]]<=1e-1 else "")
    #standard print solution
    m.print_solution()
else:
    print("Further constraints prevent the solver from finding a solution. "
                                                "Solve status: ", m.get_solve_status())
    #m.print_solution()   would generate and exception!